/***************************************************************
 *                 Mathematical Object Library                 *
 *       Abaqus Conversion tools Library - Include File        *
 *                    simula.plus@cemes.fr                     *
 *	             GNU/linux version 3.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2007,2008,2009,2012 COLLARD Christophe
 * copyright © 2007,2008,2009,2012 Centre National de la Recherche Scientifique
 * copyright © 2007,2008,2009 Arts et Métiers ParisTech
 * copyright © 2007 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2007,2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2007 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace abaqus
    \brief Simula+ Object Libraries for \htmlonly &#169; \endhtmlonly Abaqus software
*/

/*! \class abaqus::abaqus_conversion_tools
    \brief Simula+ mathematical conversion tools library for \htmlonly &#169; \endhtmlonly Abaqus software \n

    \htmlonly 
    <FONT color="#838383">

    abaqus_conversion_tools belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2007, 2008, 2009, 2012 Christophe COLLARD \n
            copyright \htmlonly &#169; 2007, 2008, 2009, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2007, 2008, 2009, 2012 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2007 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	    copyright \htmlonly &#169; 2007, 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2007 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	    copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.4.0
    \date 2007-2019
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type abaqus conversion tools
#endif

#ifndef __ABAQUS_conversion_tools_hpp
#define __ABAQUS_conversion_tools_hpp

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __parameters_h
#include "parameters.h"
#endif

#ifndef __vectors_hpp
#include "vectors.hpp"
#endif

#ifndef __tensors2_hpp
#include "tensors2.hpp"
#endif

#ifndef __symtensors2_hpp
#include "symtensors2.hpp"
#endif

#ifndef __tensors4_hpp
#include "tensors4.hpp"
#endif

using namespace std;
using namespace mol;

namespace abaqus
{


//===========================
class abaqus_conversion_tools
//===========================
{
  public :
    template <class T> friend symtensor2<T> vector2stress    (const T*);
    template <class T> friend symtensor2<T> vector2strain    (const T*);
    template <class T> friend void          stress2vector    (const symtensor2<T>&, T*);
    template <class T> friend void          stress2vector    (const symtensor2<T>&, T*);
    template <class T> friend void          strain2vector    (const tensor2<T>&, T*);
    template <class T> friend void          strain2vector    (const tensor2<T>&, T*);
    template <class T> friend tensor4<T>    matrix2symtensor (const matrix<T>&);
    template <class T> friend matrix<T>     symtensor2matrix (const tensor4<T>&);
};


//=====Public methods for abaqus conversion tools============================


/*!
  \brief Converts double* into a \f$ 2^\text{nd} \f$ order tensor for abaqus stress tensor

  \f$ \Sigma = \begin{pmatrix} v_1 & v_4 & v_5 \\ v_4 & v_2 & v_6 \\ v_5 & v_6 & v_3 \end{pmatrix} \f$
  \param v abaqus stress tensor
  \return \f$ \Sigma \f$
*/

//---------------------------------------------------------
template <class T> symtensor2<T> vector2stress (const T* v)
//---------------------------------------------------------
{
  symtensor2<T> stress(3,false);

  for (int i=1; i<=3; i++)
    { stress (i,i) = v[i-1];
      for (int j=i+1; j<=3; j++)
	stress(i,j) = v[i+j];
    }

  return stress;
}


/*!
  \brief Converts double* into a \f$ 2^\text{nd} \f$ order tensor for abaqus strain tensor

  \f$ E = \begin{pmatrix} v_1 & v_4 & v_5 \\ v_4 & v_2 & v_6 \\ v_5 & v_6 & v_3 \end{pmatrix} \f$
  \param v abaqus strain tensor
  \return \f$ E \f$
*/

//---------------------------------------------------------
template <class T> symtensor2<T> vector2strain (const T* v)
//---------------------------------------------------------
{
  symtensor2<T> strain(3,false);

  for (int i=1; i<=3; i++)
    { strain (i,i) = v[i-1];
      for (int j=i+1; j<=3; j++)
	strain(i,j) = 0.5 * v[i+j];
    }

  return strain;
}


/*!
  \brief Converts a \f$ 2^\text{nd} \f$ order tensor into a double* for abaqus stress tensor

  \f$ v = \begin{pmatrix} \sigma_{11} \\ \sigma_{22} \\ \sigma_{33} \\ \tau_{12} = \sigma_{12} \\ \tau_{13} = \sigma_{13} \\ \tau_{23} = \sigma_{23} \end{pmatrix} \f$
  \param stress \f$ \Sigma = (\sigma_{ij}) \f$
  \param *v abaqus stress tensor
*/

//-----------------------------------------------------------------------
template <class T> void stress2vector (const symtensor2<T>& stress, T* v)
//-----------------------------------------------------------------------
{
  for (int i=1; i<=3; i++)
    { v[i-1] = stress (i,i);
      for (int j=i+1; j<=3; j++)
	v[i+j] = stress(i,j);
    }
}


/*!
  \brief Converts a \f$ 2^\text{nd} \f$ order tensor into a double* for abaqus stress tensor

  \f$ v = \begin{pmatrix} \sigma_{11} \\ \sigma_{22} \\ \sigma_{33} \\ \tau_{12} = \sigma_{12} \\ \tau_{13} = \sigma_{13} \\ \tau_{23} = \sigma_{23} \end{pmatrix} \f$
  \param stress \f$ \Sigma = (\sigma_{ij}) \f$
  \param *v abaqus stress tensor
*/

//--------------------------------------------------------------------
template <class T> void stress2vector (const tensor2<T>& stress, T* v)
//--------------------------------------------------------------------
{
  for (int i=1; i<=3; i++)
    { v[i-1] = stress (i,i);
      for (int j=i+1; j<=3; j++)
	v[i+j] = 0.5 * (stress(i,j) + stress (j,i));
    }
}


/*!
  \brief Converts a \f$ 2^\text{nd} \f$ order tensor into a double* for abaqus strain tensor

  \f$ E = \begin{pmatrix} \EuScript{E}_{11} \\ \EuScript{E}_{22} \\ \EuScript{E}_{33} \\ \gamma_{12} = \EuScript{E}_{12} + \EuScript{E}_{21} \\ \gamma_{13} = \EuScript{E}_{13} + \EuScript{E}_{31} \\ \gamma_{23} = \EuScript{E}_{23} + \EuScript{E}_{32} \end{pmatrix} \f$
  \param strain \f$ E = (\EuScript{E}_{ij}) \f$
  \param *v abaqus stress tensor
*/

//-----------------------------------------------------------------------
template <class T> void strain2vector (const symtensor2<T>& strain, T* v)
//-----------------------------------------------------------------------
{
  for (int i=1; i<=3; i++)
    { v[i-1] = strain (i,i);
      for (int j=i+1; j<=3; j++)
	v[i+j] =  2 * strain(i,j);
    }
}


/*!
  \brief Converts a \f$ 2^\text{nd} \f$ order tensor into a double* for abaqus strain tensor

  \f$ E = \begin{pmatrix} \EuScript{E}_{11} \\ \EuScript{E}_{22} \\ \EuScript{E}_{33} \\ \gamma_{12} = \EuScript{E}_{12} + \EuScript{E}_{21} \\ \gamma_{13} = \EuScript{E}_{13} + \EuScript{E}_{31} \\ \gamma_{23} = \EuScript{E}_{23} + \EuScript{E}_{32} \end{pmatrix} \f$
  \param strain \f$ E = (\EuScript{E}_{ij}) \f$
  \param *v abaqus stress tensor
*/

//--------------------------------------------------------------------
template <class T> void strain2vector (const tensor2<T>& strain, T* v)
//--------------------------------------------------------------------
{
  for (int i=1; i<=3; i++)
    { v[i-1] = strain (i,i);
      for (int j=i+1; j<=3; j++)
	v[i+j] =  strain(i,j) + strain(j,i);
    }
}


/*!
  \brief Converts a matrix into symmetric \f$ 4^\text{th} \f$ order tensor.

  Let \f$ M \f$ be a matrix in  \f$ \mathbb{R}^3 \times \mathbb{R}^3 \f$. We define \f$ C \f$ the \f$ 4^\text{th} \f$ order tensor in \f$ \mathbb{R}^3 \times \mathbb{R}^3 \times \mathbb{R}^3 \times \mathbb{R}^3 \f$ such as \f$ C_{ijkl} = C_{jikl} = C_{ijlk} \f$ and
  \f$ C_{ij} = M_{ij} \f$, where we use the following notation for \f$ C \f$ indices \n
  \f$ \begin{array}{| c | c | c | c | c | c | c | c |}
  \hline C_{pq} : p \equiv & 1 & 2 & \dots & n & n+1 & n+2 & \dots \\
  \hline C_{ijkl} : ij \equiv & 11 & 22 & \dots & nn & 12 & 13 & \dots \\
  \hline
  \end{array}
  \f$
  \param mat matrix M
  \return C
  \n\n
  \f$ \textbf{Example :} \text{ in dimension 3 - elasticity constitutive equation} \f$ \n
  \f$ \begin{pmatrix}
      \sigma_{11} \\ \sigma_{22} \\ \sigma_{33} \\ \sigma_{12} \\ \sigma_{13} \\ \sigma_{23}
      \end{pmatrix}
      =
      \begin{pmatrix}
      C_{1111} & C_{1122} & C_{1133} & C_{1112} & C_{1113} & C_{1123} \\
      C_{2211} & C_{2222} & C_{2233} & C_{2212} & C_{2213} & C_{2223} \\
      C_{3311} & C_{3322} & C_{3333} & C_{3312} & C_{3313} & C_{3323} \\
      C_{1211} & C_{1222} & C_{1233} & C_{1212} & C_{1213} & C_{1223} \\
      C_{1311} & C_{1322} & C_{1333} & C_{1312} & C_{1313} & C_{1323} \\
      C_{2311} & C_{2322} & C_{2333} & C_{2312} & C_{2313} & C_{2323} \\
      \end{pmatrix}
      \begin{pmatrix}
      \EuScript{E}_{11} \\ \EuScript{E}_{22} \\ \EuScript{E}_{33} \\ 2 \EuScript{E}_{12} \\ 2 \EuScript{E}_{13} \\ 2 \EuScript{E}_{23}
      \end{pmatrix}
  \f$ \n
*/

//-------------------------------------------------------------------
template <class T> tensor4<T> matrix2symtensor (const matrix<T>& mat)
//-------------------------------------------------------------------
{
  assert (mat.Rows() && mat.Columns());

  int nb1 = .5 * (sqrt(1.+8*mat.Rows()) - 1);
  int nb2 = .5 * (sqrt(1.+8*mat.Columns()) - 1);
  tensor4<T> tsr (nb1, nb1, nb2, nb2, false);

  for (int i=1,p=nb1; i<=nb1; i++)
    for (int j=i,q=nb2; j<=nb1; j++,q=nb2)
      { if (i!=j) p++;
	for (int k=1; k<=nb2; k++)
	  for (int l=k; l<=nb2; l++)
	    if (i==j && k==l) tsr(i,i,k,k) = mat(i,k);
	    else if (i==j) tsr(i,i,k,l) = tsr(i,i,l,k) = mat(i,++q);
	         else if (k==l) tsr(i,j,k,k) = tsr(j,i,k,k) = mat(p,k);
		      else tsr(i,j,k,l) = tsr(j,i,k,l) = tsr(i,j,l,k) = tsr(j,i,l,k) = mat(p,++q);
      }

  return tsr;
}


/*!
  \brief Converts a symmetric \f$ 4^\text{th} \f$ order tensor into a matrix.

  Let \f$ C \f$ be a \f$ 4^\text{th} \f$ order tensor in \f$ \mathbb{R}^3 \times \mathbb{R}^3 \times \mathbb{R}^3 \times \mathbb{R}^3 \f$ such as \f$ C_{ijkl} = C_{jikl} = C_{ijlk} \f$.
  We define the matrix \f$ M \in \mathbb{R}^3 \times \mathbb{R}^3 \f$ such as \n
  \f$ M_{ij} = C_{ij} \f$, where we use the following notation for \f$ C \f$ indices \n
  \f$ \begin{array}{| c | c | c | c | c | c | c | c |}
  \hline C_{pq} : p \equiv & 1 & 2 & \dots & n & n+1 & n+2 & \dots \\
  \hline C_{ijkl} : ij \equiv & 11 & 22 & \dots & nn & 12 & 13 & \dots \\
  \hline
  \end{array}
  \f$
  \param tsr \f$ 4^\text{th} \f$ order tensor C
  \return M
  \n\n
  \f$ \textbf{Example :} \text{ in dimension 3 - elasticity constitutive equation} \f$ \n
  \f$ \begin{pmatrix}
      \sigma_{11} \\ \sigma_{22} \\ \sigma_{33} \\ \sigma_{12} \\ \sigma_{13} \\ \sigma_{23}
      \end{pmatrix}
      =
      \begin{pmatrix}
      C_{1111} & C_{1122} & C_{1133} & C_{1112} & C_{1113} & C_{1123} \\
      C_{2211} & C_{2222} & C_{2233} & C_{2212} & C_{2213} & C_{2223} \\
      C_{3311} & C_{3322} & C_{3333} & C_{3312} & C_{3313} & C_{3323} \\
      C_{1211} & C_{1222} & C_{1233} & C_{1212} & C_{1213} & C_{1223} \\
      C_{1311} & C_{1322} & C_{1333} & C_{1312} & C_{1313} & C_{1323} \\
      C_{2311} & C_{2322} & C_{2333} & C_{2312} & C_{2313} & C_{2323} \\
      \end{pmatrix}
      \begin{pmatrix}
      \EuScript{E}_{11} \\ \EuScript{E}_{22} \\ \EuScript{E}_{33} \\ 2 \EuScript{E}_{12} \\ 2 \EuScript{E}_{13} \\ 2 \EuScript{E}_{23}
      \end{pmatrix}
  \f$ \n
*/

//-------------------------------------------------------------------
template <class T> matrix<T> symtensor2matrix (const tensor4<T>& tsr)
//-------------------------------------------------------------------
{
  assert (tsr.dim1()==tsr.dim2() && tsr.dim3()==tsr.dim4());

  int nb1 = tsr.dim1();
  int nb2 = tsr.dim3();
  matrix<T> mat (.5*nb1*(nb1+1), .5*nb2*(nb2+1), false);

  for (int i=1,p=nb1; i<=nb1; i++)
    for (int j=i,q=nb2; j<=nb1; j++,q=nb2)
      { if (i!=j) p++;
	for (int k=1; k<=nb2; k++)
	  for (int l=k; l<=nb2; l++)
	    { assert ( relative_error (tsr(i,j,k,l), tsr(i,j,l,k)) );
	      assert ( relative_error (tsr(i,j,k,l), tsr(j,i,k,l)) );
	      assert ( relative_error (tsr(i,j,k,l), tsr(j,i,l,k)) );
	      if (i==j && k==l) mat (i,k) = tsr(i,j,k,l);
	      else if (i==j) mat(i,++q) = tsr(i,j,k,l);
	           else if (k==l) mat(p,k) = tsr(i,j,k,l);
		        else mat(p,++q) = tsr(i,j,k,l);
	    }
      }

  return mat;
}


}


#endif
